' SAP Complete Tool Workflow Script
' Author: Dario Pascoal
'
' Description: This script handles the complete tool processing workflow in SAP by executing
' all necessary operations within a single ZLIFECYCLE transaction. It combines three major
' components: plant/storage location checking for routing decisions, tool information extraction,
' and comprehensive dates/intervals extraction (calibration, certification, control, and NEN3140 data).
'
' The script works by first connecting to an active SAP session, then performing a plant check
' to determine the appropriate workflow routing. If the routing indicates normal processing
' should continue, it extracts detailed tool information including serial numbers, equipment
' numbers, and material data. Finally, it retrieves all relevant dates and intervals from
' the classification system for maintenance scheduling purposes.
'
' Prerequisites: Active SAP GUI session with appropriate user permissions for ZLIFECYCLE transaction
' Parameters: 
'   - Argument 1: Barcode ID (required) - The tool barcode to process
'   - Argument 2: Log file path (optional) - Path for detailed logging output
' Returns: Structured output with plant check results, routing decisions, tool data, and dates/intervals
'
On Error Resume Next

' Set scripting mode to true to enable SAP GUI automation
Set WSHShell = CreateObject("WScript.Shell")

' Disable SAP GUI scripting security warnings in Windows registry
' These settings prevent popup dialogs that would interrupt automation
' Each registry setting controls a different type of security warning
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' Declare variables for logging functionality
' These variables handle file-based logging for debugging and audit purposes
Dim logFile, logFilePath

' Get user's Downloads folder path from Windows environment variables
' This provides a standard location that all users have access to
Set WShell = CreateObject("WScript.Shell")
DownloadsPath = WShell.ExpandEnvironmentStrings("%USERPROFILE%") & "\Downloads\"

' Extract barcode ID from command line arguments
' This is the primary identifier for the tool we're processing
Dim barcodeId
barcodeId = WScript.Arguments.Item(0)

' Initialize logging system if log file path is provided as second argument
' This allows optional detailed logging for troubleshooting and audit trails
If WScript.Arguments.Count > 1 Then
    logFilePath = WScript.Arguments.Item(1)
    ' Attempt to open the log file in append mode (mode 8 = append)
    ' This preserves existing log entries while adding new ones
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    ' Handle potential file access errors gracefully
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Create a clear log entry header for this script execution
        LogToFile "========================================="
        LogToFile "Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Barcode ID: " & barcodeId
        LogToFile "========================================="
    End If
End If

' Establish connection to SAP GUI application
' This connects to an existing SAP session that must already be running
' The connection hierarchy is: Application -> Connection -> Session
If Not IsObject(application) Then
    Set SapGuiAuto = GetObject("SAPGUI")
    Set application = SapGuiAuto.GetScriptingEngine
End If
If Not IsObject(connection) Then
    Set connection = application.Children(0)
End If
If Not IsObject(session) Then
    Set session = connection.Children(0)
End If
' Enable event handling for SAP GUI automation
If IsObject(WScript) Then
    WScript.ConnectObject session, "on"
    WScript.ConnectObject application, "on"
End If

' Handle potential SAP GUI security dialog automatically
' This function deals with security warnings that may appear during automation
' It checks for the security dialog window and automatically accepts it
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    ' Loop for up to 10 seconds looking for security dialog
    Do While timeout < 10
        ' Check if SAP GUI Security dialog window is present
        If shell.AppActivate("SAP GUI Security") Then
            WScript.Sleep 500
            shell.SendKeys " "  ' Press space to check "Remember my decision" checkbox
            WScript.Sleep 200
            shell.SendKeys "{ENTER}"  ' Press Enter to click Allow button
            dialogFound = True
            Exit Do
        End If
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' Execute security dialog handler
HandleSecurityDialog()

' Launch the ZLIFECYCLE SAP transaction
' This transaction is used for tool lifecycle management and contains all the data we need
session.findById("wnd[0]").maximize
session.findById("wnd[0]/tbar[0]/okcd").text = "zlifecycle"
session.findById("wnd[0]").sendVKey 0
WScript.Sleep 2000

' Input the barcode ID into the search field and execute the search
' This will bring up all tools matching the provided barcode
session.findById("wnd[0]/usr/txtP_BARC").text = barcodeId
session.findById("wnd[0]/usr/txtP_BARC").caretPosition = Len(barcodeId)
session.findById("wnd[0]").sendVKey 8
WScript.Sleep 2000

' ============================================================
' PART 1: PLANT CHECK LOGIC
' This section implements business rules for routing tools based on 
' plant and storage location combinations. Different combinations
' require different processing workflows in the DHL system.
' ============================================================

' Initialize variables for plant and storage location data extraction
Dim grid, rowCount, rowIndex, recPlant, slocTo
Set grid = session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell")

recPlant = ""
slocTo = ""

' Extract plant and storage location data from the SAP grid
' The grid contains multiple rows, we need to find the first row with valid data
If Not grid Is Nothing Then
    rowCount = grid.RowCount

    ' Loop through all rows in the grid to find plant and storage location data
    For rowIndex = 0 To rowCount - 1
        ' UMWRK = Receiving Plant field, UMLGO = Storage Location field
        recPlant = Trim(grid.GetCellValue(rowIndex, "UMWRK"))
        slocTo   = Trim(grid.GetCellValue(rowIndex, "UMLGO"))

        ' Stop at the first row that has both values populated
        If recPlant <> "" And slocTo <> "" Then
            Exit For
        End If
    Next
End If

' Output plant check results in a structured format for the calling application
LogMessage "PLANT_CHECK:RecPlant:" & recPlant
LogMessage "PLANT_CHECK:SlocTo:" & slocTo

' Apply business logic to determine the appropriate workflow routing
' These rules are based on DHL's operational requirements:
' - NL10 + D008: Normal automated processing can continue
' - NL10 + other: Requires manual team leader intervention
' - NL05 + D008: Must use TMM web form instead of this system
' - NL05 + other: Requires manual team leader intervention
' - Any other combination: Default to team leader intervention for safety
Dim routingResult
If (recPlant = "NL10" And slocTo = "D008") Then
    routingResult = "CONTINUE_NORMAL"
ElseIf (recPlant = "NL10" And slocTo <> "D008") Then
    routingResult = "GO_TO_TEAM_LEADER"
ElseIf (recPlant = "NL05" And slocTo = "D008") Then
    routingResult = "USE_TMM_WEBFORM"
ElseIf (recPlant = "NL05" And slocTo <> "D008") Then
    routingResult = "GO_TO_TEAM_LEADER"
Else
    routingResult = "GO_TO_TEAM_LEADER"
End If

LogMessage "ROUTING_RESULT:" & routingResult

' ============================================================
' ROUTING DECISION IMPLEMENTATION
' Only continue with automated processing if routing indicates normal workflow
' All other routing results require manual intervention or alternative systems
' ============================================================

If routingResult <> "CONTINUE_NORMAL" Then
    ' Exit the ZLIFECYCLE transaction and return to main SAP menu
    ' This leaves the user at a clean starting point for manual processing
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    
    ' Close log file properly and terminate script execution
    CloseLogFile()
    WScript.Quit
End If

' ============================================================
' PART 2: TOOL INFORMATION EXTRACTION
' Since routing approved normal processing, extract detailed tool data
' This remains within the same ZLIFECYCLE transaction for efficiency
' ============================================================

' Select the first tool record from the search results grid
' This focuses SAP on the specific tool we want to process
session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell").currentCellColumn = "SERNR"
session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell").selectedRows = "0"
session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell").clickCurrentCell

' Check equipment type to identify stepper parts (type "U")
' Stepper parts require special handling and cannot be processed through normal workflow
session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/ctxtITOB-EQTYP").setFocus
equipmentType = session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/ctxtITOB-EQTYP").text

' Handle stepper parts with early exit and special error code
If equipmentType = "U" Then
    ' Return a special error code to indicate stepper part detection
    ' This allows the calling application to handle stepper parts differently
    LogMessage "ERROR:STEPPER_PART"
    
    ' Navigate back to the ZLIFECYCLE search screen for user convenience
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    
    ' Clean up and exit
    CloseLogFile()
    WScript.Quit
End If

' Extract Equipment ID directly from the grid data
' This is more reliable than navigating through form fields
On Error Resume Next
equipmentID = session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell").getCellValue(0, "EQUNR")
On Error GoTo 0

' Extract tool description from the equipment details form
' The description provides human-readable information about the tool's purpose
session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152B:SAPLITO0:1530/txtITOB-SHTXT").setFocus
description = session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152B:SAPLITO0:1530/txtITOB-SHTXT").text

' Extract Material Number - this identifies the tool type in the inventory system
session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/ctxtITOB-MATNR").setFocus
materialNumber = session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/ctxtITOB-MATNR").text

' Extract Serial Number - this is the unique identifier for this specific tool instance
session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/txtITOB-SERNR").setFocus
serialNumber = session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/txtITOB-SERNR").text

' Navigate to tab 06 to access Equipment Number field
' The SAP interface uses multiple tabs to organize different types of equipment data
session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\06").select
WScript.Sleep 500

' Extract Equipment Number from tab 06
' This number is used for internal SAP equipment tracking
session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\06/ssubSUB_DATA:SAPLITO0:0122/subSUB_0122A:SAPLITO0:1520/ctxtITOBATTR-EQUNR").setFocus
equipmentNumber = session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\06/ssubSUB_DATA:SAPLITO0:0122/subSUB_0122A:SAPLITO0:1520/ctxtITOBATTR-EQUNR").text

' Navigate to tab 07 to access Barcode ID field
' The barcode ID is stored in a custom field on this tab
session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\07").select

' Extract Barcode ID from tab 07 and verify it matches our search criteria
' This provides confirmation that we're processing the correct tool
session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\07/ssubSUB_DATA:SAPLITO0:0102/subSUB_0102B:SAPLITO0:1080/ssubXUSR1080:SAPLXTOB:1003/txtITOB-ZZBARCODE").setFocus
barcodeIdFromSAP = session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\07/ssubSUB_DATA:SAPLITO0:0102/subSUB_0102B:SAPLITO0:1080/ssubXUSR1080:SAPLXTOB:1003/txtITOB-ZZBARCODE").text

' ============================================================
' PART 3: DATES AND INTERVALS EXTRACTION
' Extract comprehensive maintenance scheduling data while remaining
' in the same ZLIFECYCLE transaction. This includes all critical
' dates and intervals for calibration, certification, control, and NEN3140 compliance.
' ============================================================

' Define the complete list of fields to extract from SAP classification system
' Each array element contains: [SAP field name, Display name for output]
' These fields represent different types of maintenance requirements and their intervals
Dim labels, values, dates_i, dates_f
labels = Array( _
    Array("Next Calibration Date", "Calibration Date"), _
    Array("Next Certification Date", "Certification Date"), _
    Array("Next Control Date", "Control Date"), _
    Array("Next NEN3140 Control Date", "NEN3140 Control Date"), _
    Array("Calibration Interval", "Calibration Interval"), _
    Array("Certification Interval", "Certification Interval"), _
    Array("Control/Inspection Interval", "Control/Inspection Interval"), _
    Array("NEN3140 Control Interval", "NEN3140 Control Interval") _
)

' Access the SAP Class Overview functionality
' This contains the classification data with maintenance dates and intervals
LogMessage "Opening Class Overview..."
session.FindById("wnd[0]/tbar[1]/btn[20]").Press

' Verify that classification data is available for this equipment
' Some tools may not have classification data, which means no maintenance dates are defined
On Error Resume Next
Dim classificationAvailable
classificationAvailable = True
session.FindById("wnd[0]/usr/subSUBSCR_BEWERT:SAPLCTMS:5000/tabsTABSTRIP_CHAR/tabpTAB1/ssubTABSTRIP_CHAR_GR:SAPLCTMS:5100/tblSAPLCTMSCHARS_S")
If Err.Number <> 0 Then
    classificationAvailable = False
    LogMessage "WARNING: Classification data not available for this equipment"
End If
On Error GoTo 0

' Process classification data if available
If classificationAvailable Then
    ' Create a dictionary to store all extracted date and interval values
    ' This allows us to organize the data efficiently for output
    Set values = CreateObject("Scripting.Dictionary")    ' Iterate through each field we need to extract from the classification system
    ' For each field, we use SAP's Position function to locate it in the grid
    For dates_f = 0 To UBound(labels)
        Dim fieldLabel, displayName, dates_grid, foundValue
        fieldLabel = labels(dates_f)(0)    ' The exact field name as it appears in SAP
        displayName = labels(dates_f)(1)   ' The friendly name for our output
        
        LogMessage "Searching for field: " & fieldLabel

        ' Use SAP's Position button to navigate to the specific field in the grid
        ' This is necessary because the grid may contain many fields and we need to scroll to find ours
        On Error Resume Next
        session.FindById("wnd[0]/usr/subSUBSCR_BEWERT:SAPLCTMS:5000/tabsTABSTRIP_CHAR/tabpTAB1/ssubTABSTRIP_CHAR_GR:SAPLCTMS:5100/btnRCTMS-AUFS").Press
        
        ' Handle cases where the Position button is not accessible
        If Err.Number <> 0 Then
            LogMessage "WARNING: Could not access Position button for field: " & fieldLabel
            foundValue = "Not Found"
            values.Add displayName, foundValue
            Err.Clear
        Else
            On Error GoTo 0
            
            ' Enter the field name in the search dialog and execute the search
            session.FindById("wnd[1]/usr/txtCLHP-CR_STATUS_TEXT").Text = fieldLabel
            session.FindById("wnd[1]").SendVKey 0
            ' Access the classification grid to search for our field
            ' The grid contains rows of characteristic names and their values
            On Error Resume Next
            Set dates_grid = session.FindById("wnd[0]/usr/subSUBSCR_BEWERT:SAPLCTMS:5000/tabsTABSTRIP_CHAR/tabpTAB1/ssubTABSTRIP_CHAR_GR:SAPLCTMS:5100/tblSAPLCTMSCHARS_S")
            
            ' Handle cases where the grid is not accessible
            If Err.Number <> 0 Then
                LogMessage "WARNING: Could not access grid for field: " & fieldLabel
                foundValue = "Not Found"
                values.Add displayName, foundValue
                Err.Clear
            Else
                On Error GoTo 0
                foundValue = ""
                
                ' Scan through all rows in the grid to find our specific field
                ' Column 0 contains field names, Column 1 contains the values
                For dates_i = 0 To dates_grid.Rows.Count - 1
                    On Error Resume Next
                    Dim cellValue
                    cellValue = dates_grid.GetCell(dates_i, 0).Text
                    
                    ' Check if this row contains the field we're looking for
                    If Err.Number = 0 And cellValue = fieldLabel Then
                        ' Extract the value from column 1 of the matching row
                        foundValue = dates_grid.GetCell(dates_i, 1).Text
                        If Err.Number = 0 Then
                            ' Handle empty values appropriately
                            If foundValue = "" Then
                                LogMessage "Found " & fieldLabel & " but value is empty"
                                foundValue = "N/A"
                            Else
                                LogMessage "Found " & fieldLabel & ": " & foundValue
                            End If
                            Exit For
                        End If
                    End If
                    Err.Clear
                    On Error GoTo 0
                Next

                ' Mark field as not found if we didn't locate it in the grid
                If foundValue = "" Then
                    foundValue = "Not Found"
                    LogMessage "Field not found in grid: " & fieldLabel
                End If

                ' Store the result in our dictionary for later output
                values.Add displayName, foundValue
            End If
        End If
    Next
    
    ' Generate structured output for all extracted dates and intervals
    ' This creates a standardized format that the calling application can parse
    LogMessage "Building dates and intervals output results..."
    For Each displayName In values.Keys
        ' Output format: DATE_FIELD:DisplayName:Value
        ' This allows the calling application to distinguish between different types of data
        LogMessage "DATE_FIELD:" & displayName & ":" & values(displayName)
    Next
Else
    ' Handle equipment that doesn't have classification data
    ' This informs the calling application that no maintenance scheduling data is available
    LogMessage "DATE_DATA_INFO:No classification data available"
End If

' Navigate back through SAP screens to return to the main ZLIFECYCLE interface
' This ensures the user is left in a clean state for subsequent operations
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press

' Output all extracted tool information in a structured format
' This provides the calling application with all the essential tool data
LogMessage "SAP_FIELD:Description:" & description
LogMessage "SAP_FIELD:SerialNumber:" & serialNumber
LogMessage "SAP_FIELD:EquipmentNumber:" & equipmentNumber
LogMessage "SAP_FIELD:BarcodeID:" & barcodeIdFromSAP
LogMessage "SAP_FIELD:MaterialNumber:" & materialNumber
LogMessage "SAP_FIELD:EquipmentType:" & equipmentType

' ============================================================
' HELPER FUNCTIONS
' These functions provide logging and cleanup functionality
' to support debugging and proper resource management
' ============================================================

' Function to write log messages to file only
' This is used for detailed logging that doesn't need to appear in console output
' Parameters: message - The text to write to the log file
Sub LogToFile(message)
    On Error Resume Next
    If Not logFile Is Nothing Then
        ' Write timestamp, script name, and message to log file
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Function to output messages to both console and log file
' This ensures important information is visible to both user and log
' Parameters: message - The text to output to console and write to log
Sub LogMessage(message)
    WScript.Echo message    ' Display on console for immediate feedback
    LogToFile message       ' Write to log file for permanent record
End Sub

' Function to properly close the log file and clean up resources
' This should be called before script termination to ensure log integrity
Sub CloseLogFile()
    On Error Resume Next
    If Not logFile Is Nothing Then
        ' Write script completion timestamp and separator
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        ' Close file handle and release object reference
        logFile.Close
        Set logFile = Nothing
    End If
End Sub

' Perform final cleanup before script termination
' This ensures all resources are properly released
CloseLogFile()
